#!/bin/bash

# Written 2009 by  Eric Hameleers, Eindhoven, NL

# Convert 32bit slackware packages 'en masse' into compatibility packages
# to be installed on slackware64 - providing the 32bit part of multilib.
# The string "-compat32" is added at the end of package name when a
# compatibility package gets created.  This allows it to be installed
# on slackware64 alongside the native 64bit versions.
# For example: the original 32bit package "bzip2" will be converted to a new
#              package with the name "bzip2-compat32"

# You also have to install multilib versions of glibc and gcc !

# Before we start
[ -x /bin/id ] && CMD_ID="/bin/id" || CMD_ID="/usr/bin/id"
if [ "$($CMD_ID -u)" != "0" ]; then
  echo "You need to be root to run $(basename $0)."
  exit 1
fi

# Should we be verbose?
VERBOSE=${VERBOSE:-1}

# Helpful instructions in case the user asks for it:
function show_help () {
  # Write the help text to output:
  cat <<EOF

Usage: $0 -i 32bit_package_tree  [-d output_directory]

$(basename $0) converts an essential subset of 32-bit Slackware
packages into 'compatibility' packages for 64-bit Slackware.

required parameters::
  -i 32bit_package_tree        A 32bit Slackware package-tree. It should have
                               the a,ap,d,..,y directories immediately below.
optional parameters::
  -d destination_directory     create packages in this directory.
                               By default, the new packages will be created
                               in your current directory.

EOF
}

# A function to retrieve the fullname of a package:
function get_pkgfullpath () {
  local IP="$1"
  for FP in $(ls $IP*.t?z) ; do
    if [ "$(echo $FP|rev|cut -d- -f4-|cut -d/ -f1| rev)" = "$(basename $IP)" ];
    then
      echo $FP
      continue
    fi
  done
}

# Parse the commandline parameters:
while [ ! -z "$1" ]; do
  case $1 in
    -d|--destdir)
      TARGET64ROOT="$(cd $(dirname ${2}); pwd)/$(basename ${2})"
      shift 2
      ;;
    -h|--help)
      show_help
      exit 0
      ;;
    -i|--inputdir)
      SLACK32ROOT="$(cd $(dirname ${2}); pwd)/$(basename ${2})"
      shift 2
      ;;
    -*)
      echo "Unsupported parameter '$1'!"
      exit 1
      ;;
    *)
      # Do nothing
      shift
      ;;
  esac
done

# The root directory of 32bit Slackware packages
# (should have the a,ap,d,..,y directories immediately below):
# Let's use a fallback directory in case I am lazy:
SLACK32ROOT=${SLACK32ROOT:-"~ftp/pub/Linux/Slackware/slackware-current/slackware/"}

# The output directory for our converted packages; defaults to the current dir.
# Note that {a,ap,d,l,n,x}-compat32 directories will be created below this
# directory if they do not yet exist:
TARGET64ROOT=${TARGET64ROOT:-"$(pwd)"}

# Where the scripts are:
SRCDIR=$(cd $(dirname $0); pwd)

# The script that does the package conversion:
CONV32=$SRCDIR/convertpkg-compat32

# Bail out if the conversion script is not available/working:
if [ ! -f $CONV32 ]; then
  echo "Required script '$CONV32' is not present or not executable! Aborting..."
  exit 1
fi

# We can not proceed if there are no packages:
if [ ! -d $SLACK32ROOT/a -o ! -d $SLACK32ROOT/ap -o ! -d $SLACK32ROOT/d -o ! -d $SLACK32ROOT/l -o ! -d $SLACK32ROOT/n -o ! -d $SLACK32ROOT/x ]; then
  echo "Required package directories a,ap,d,l,n,x below '$SLACK32ROOT' are not found! Aborting..."
  exit 1
fi

# If a destination_directory was specified, abort now if we can not create it:
if [ -n "$TARGET64ROOT" -a ! -d "$TARGET64ROOT" ]; then
  echo "Creating output directory '$TARGET64ROOT'..."
  mkdir -p $TARGET64ROOT
  if [ ! -w "$TARGET64ROOT" ]; then
    echo "Creation of output directory '$TARGET64ROOT' failed!"
    exit 3
  fi
fi

# This is the script's internal list of what I consider as the essential
# 32bit multilib package set for your Slackware64:

# The A/ series:
A_COMPAT32="
bzip2
cups
cxxlibs
dbus
e2fsprogs
openssl-solibs
util-linux-ng
"

# The AP/ series:
AP_COMPAT32="
mpg123
mysql
"

# The D/ series:
D_COMPAT32="
libtool
"

# The L/ series:
L_COMPAT32="
alsa-lib
alsa-oss
atk
audiofile
cairo
dbus-glib
esound
expat
freetype
gamin
glib2
gtk+2
hal
jasper
lcms
libart_lgpl
libexif
libgphoto2
libjpeg
libidn
libmng
libpng
libtermcap
libtiff
libv4l
libxml2
libxslt
ncurses
pango
popt
qt
readline
sdl
seamonkey-solibs
svgalib
zlib
"

# The N/ series:
N_COMPAT32="
curl
cyrus-sasl
gnutls
libgcrypt
libgpg-error
openldap-client
openssl
"

# The X/ series:
X_COMPAT32="
fontconfig
libFS
libICE
libSM
libX11
libXScrnSaver
libXTrap
libXau
libXaw
libXcomposite
libXcursor
libXdamage
libXdmcp
libXevie
libXext
libXfixes
libXfont
libXfontcache
libXft
libXi
libXinerama
libXmu
libXp
libXpm
libXprintUtil
libXrandr
libXrender
libXres
libXt
libXtst
libXv
libXvMC
libXxf86dga
libXxf86misc
libXxf86vm
libdmx
libdrm
libfontenc
liblbxutil
libxcb
mesa
pixman
"

# Create target directories if they do not exist:
for TDIR in a-compat32 ap-compat32 d-compat32 l-compat32 n-compat32 x-compat32 ; do
  mkdir -p $TARGET64ROOT/$TDIR
  if [ ! -w $TARGET64ROOT/$TDIR ]; then
    echo "Directory '$TARGET64ROOT/$TDIR' is not writable! Aborting..."
    exit 1
  fi
done

# Convert the 32bit packages from A AP D L N and X series:
echo "***"
echo "*** Starting the conversion process:"

echo "*** 'A' series:"
for INPKG in $A_COMPAT32 ; do
  FULLPKG=$(get_pkgfullpath $SLACK32ROOT/a/$INPKG)
  if [ -z "$FULLPKG" ]; then
    echo "*** FAIL: package '$INPKG' was not found!"
    continue
  fi
  [ "$VERBOSE" = "1" ] && echo "--- $INPKG"
  sh $CONV32 -i $FULLPKG -d $TARGET64ROOT/a-compat32
done

echo "*** 'AP' series:"
for INPKG in $AP_COMPAT32 ; do
  FULLPKG=$(get_pkgfullpath $SLACK32ROOT/ap/$INPKG)
  if [ -z "$FULLPKG" ]; then
    echo "*** FAIL: package '$INPKG' was not found!"
    continue
  fi
  [ "$VERBOSE" = "1" ] && echo "--- $INPKG"
  sh $CONV32 -i $FULLPKG -d $TARGET64ROOT/ap-compat32
done

echo "*** 'D' series:"
for INPKG in $D_COMPAT32 ; do
  FULLPKG=$(get_pkgfullpath $SLACK32ROOT/d/$INPKG)
  if [ -z "$FULLPKG" ]; then
    echo "*** FAIL: package '$INPKG' was not found!"
    continue
  fi
  [ "$VERBOSE" = "1" ] && echo "--- $INPKG"
  sh $CONV32 -i $FULLPKG -d $TARGET64ROOT/d-compat32
done

echo "*** 'L' series:"
for INPKG in $L_COMPAT32 ; do
  FULLPKG=$(get_pkgfullpath $SLACK32ROOT/l/$INPKG)
  if [ -z "$FULLPKG" ]; then
    echo "*** FAIL: package '$INPKG' was not found!"
    continue
  fi
  [ "$VERBOSE" = "1" ] && echo "--- $INPKG"
  sh $CONV32 -i $FULLPKG -d $TARGET64ROOT/l-compat32
done

echo "*** 'N' series:"
for INPKG in $N_COMPAT32 ; do
  FULLPKG=$(get_pkgfullpath $SLACK32ROOT/n/$INPKG)
  if [ -z "$FULLPKG" ]; then
    echo "*** FAIL: package '$INPKG' was not found!"
    continue
  fi
  [ "$VERBOSE" = "1" ] && echo "--- $INPKG"
  sh $CONV32 -i $FULLPKG -d $TARGET64ROOT/n-compat32
done

echo "*** 'X' series:"
for INPKG in $X_COMPAT32 ; do
  FULLPKG=$(get_pkgfullpath $SLACK32ROOT/x/$INPKG)
  if [ -z "$FULLPKG" ]; then
    echo "*** FAIL: package '$INPKG' was not found!"
    continue
  fi
  [ "$VERBOSE" = "1" ] && echo "--- $INPKG"
  sh $CONV32 -i $FULLPKG -d $TARGET64ROOT/x-compat32
done

echo "*** Conversion done!"
echo "***"
